<?php

namespace Nidec\LaravelOauthClient;

use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use InvalidArgumentException;

class OAuthController extends Controller
{
    public function redirectToProvider(Request $request): RedirectResponse
    {
        $request->session()->put('state', $state = Str::random(40));

        $request->session()->put(
            'code_verifier', $codeVerifier = Str::random(128)
        );

        $codeChallenge = strtr(rtrim(
            base64_encode(hash('sha256', $codeVerifier, true))
            , '='), '+/', '-_');

        $url = config('oauth_client.auth_server') . config('oauth_client.authorize_endpoint');

        $query = http_build_query([
            'client_id' => config('oauth_client.client_id'),
            'redirect_uri' => route('oauth.callback'),
            'response_type' => 'code',
            'scope' => config('oauth_client.scopes'),
            'state' => $state,
            'code_challenge' => $codeChallenge,
            'code_challenge_method' => 'S256',
        ]);

        return redirect($url . '?' . $query);
    }

    public function handleProviderCallback(Request $request): RedirectResponse
    {
        $state = $request->session()->pull('state');

        $codeVerifier = $request->session()->pull('code_verifier');

        throw_unless(
            strlen($state) > 0 && $state === $request->state,
            InvalidArgumentException::class,
            'Invalid state value.'
        );

        $url = config('oauth_client.auth_server') . config('oauth_client.token_endpoint');

        $response = Http::asForm()->post($url, [
            'grant_type' => 'authorization_code',
            'client_id' => config('oauth_client.client_id'),
            'redirect_uri' => route('oauth.callback'),
            'code_verifier' => $codeVerifier,
            'code' => $request->code,
            'scope' => config('oauth_client.scopes'),
        ]);

        abort_unless($response->ok(), 400, 'Token exchange failed');

        $tokens = $response->json();
        Session::put('access_token',  $tokens['access_token']);
        Session::put('refresh_token', $tokens['refresh_token']);
        Session::put('expires_at',    now()->addSeconds($tokens['expires_in'])->timestamp);

        return redirect('/');
    }
}
